import matplotlib as mpl
# mpl.use('pgf')
mpl.use('pdf')

import matplotlib.pyplot as plt
import numpy as np

plt.rcParams['pdf.fonttype'] = 42
plt.rcParams['ps.fonttype'] = 42

plt.rcParams.update({
        "font.family": "serif",  # use serif/main font for text elements
        "text.usetex": False,    # use inline math for ticks
        "pgf.rcfonts": False,    # don't setup fonts from rc parameters
        "pgf.texsystem": "pdflatex",
        "pgf.preamble": "\n".join([
            "\\usepackage[utf8x]{inputenc}",
            "\\usepackage[T1]{fontenc}",
            "\\usepackage{times}"
            "%\\renewcommand{\\rmdefault}{ptm}",
            "%\\renewcommand{\\sfdefault}{phv}",
        ])
    })

#####################################################################################
#############################
#############################     This code plots Figures 1-7 from 
#############################     the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."
#############################
#############################     Before running this code it is necessary to create
#############################     the corresponding .npy files by using 1. run_algorithms.py
############################# 
#####################################################################################

################# Parameter

# Number of iterations
T=100000
# Number of points on figure
nb_points=30

# small 3x3 matrix game
A = [[3,0,-3],[0,3,-4],[0,0,1]]
A_T=np.transpose(A)
n=len(A[0])
m=n
nb_last_iterate = 100

# number of points to show for Figures 4,5,6
nb_trajectory = 2000

eta = 0.1


## In case we want to show the results for only one algorithm:
    
alg = 'rmp' # Regret Matching+
# alg = 'prmp' # Predictive Regret Matching+
# alg = 'alt_rmp' # Alternating Regret Matching+
# alg = 'alt_prmp' # Alternating Predictive Regret Matching+
# alg = 'crmp' # ExRM+
# alg = 'crmp_rs' # RS-ExRM+
# alg = 'smooth_prmp' # Smooth Predictive Regret Matching+ (SPRM+)
# alg = 'smooth_prmp_rs' # RS-SPRM+
# alg = 'extragradient_alg' # Extragradient algorithm
# alg = 'optimistic_omd' # Optimistic OMD


#### Boolean to choose which Figure to plot
############## Fig 1: small matrix game
### RM+, alt RM+, PRM+, alt PRM+ on the 3x3 matrix game:
plot_small_matrix = 0
############## Fig 3: Duality gaps of
###  RM+, alt RM+, PRM+, alt PRM+, ExRM+, Smooth PRM+, EG, OG
### over Kuhn, Goofspiel, small matrix, random instances:
plot_both_matrix_instances_4_plots = 0

############## Fig 4-6: showing the last iterates of
# RM+, alt RM+, PRM+ on the small matrix instance
plot_strategy_profile_small_matrix = 0
############## Fig 7: Duality gaps of
###   ExRM+, Smooth PRM+, RS-ExRM+, RS-Smooth PRM+
### over Kuhn, Goofspiel, small matrix, random instances:
plot_both_matrix_instances_4_plots_with_restart = 0

############## Fig 1: small matrix game
### RM+, alt RM+, PRM+, alt PRM+ on the 3x3 matrix game:
if plot_small_matrix:
      
    T_mat = 100000
    nb_points_loc = 15
    x_axis = np.asarray(range(T_mat-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    fig, ax = plt.subplots()

    
    ax.tick_params(labelsize=25)
    ax.set_yscale('log')
    ax.set_xscale('log')
    
    
    result_duality_gap_rmp = np.load(f'numerical_results/rmp_duality_gap_{T_mat}.npy')        
    result_duality_gap_alt_rmp = np.load(f'numerical_results/alt_rmp_duality_gap_{T_mat}.npy')        
    result_duality_gap_prmp = np.load(f'numerical_results/prmp_duality_gap_{T_mat}.npy')   
    result_duality_gap_alt_prmp = np.load(f'numerical_results/alt_prmp_duality_gap_{T_mat}.npy')   

    ax.plot(x_axis[indices], result_duality_gap_rmp[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label = r'RM$^+$')
    ax.plot(x_axis[indices], result_duality_gap_alt_rmp[indices], color='green', marker='p', linestyle='--',linewidth=2, markersize=10, label = r'PRM$^+$')
    ax.plot(x_axis[indices], result_duality_gap_prmp[indices], color='blue', marker='D', linestyle='-',linewidth=2, markersize=10,label = r'alt. RM$^+$')
    ax.plot(x_axis[indices], result_duality_gap_alt_prmp[indices], color='black', marker='d', linestyle='-',linewidth=2, markersize=10, label = r'alt. PRM$^+$')

    plt.xscale('log')
    plt.yscale('log')
    
    ax.set_xlabel('Number of iterations',fontsize=20)
    ax.set_ylabel('Duality gap',fontsize=20)
    ax.tick_params(axis='both', which='major', labelsize=20)
    
    if T_mat == 100000:
        plt.xticks([10,100,1000,10000,100000])
    
    # Rebuttal: for same yaxis as the plot with extragradient alg + optimistic omd:
    ax.set_ylim(10e-18,5)

    ax.grid()

    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.15,0,0),fontsize=20)
    fig.savefig(f"figures/duality_gap_T_{T_mat}_all_alg_wrap.pgf", bbox_inches='tight')
    fig.savefig(f"figures/duality_gap_T_{T_mat}_all_alg_wrap.pdf", bbox_inches='tight')
        
    plt.show()

############## Fig 3: Duality gaps of
###  RM+, alt RM+, PRM+, alt PRM+, ExRM+, Smooth PRM+, EG, OG
### over Kuhn, Goofspiel, small matrix, random instances:
### Here we choose the best stepsizes in {1,10e-1,10e-2,10e-3,10e-4} for each algorithm
if plot_both_matrix_instances_4_plots:
  
   
    # plot results for 3x3 matrix instances and for random matrix games
    # The random simulations only go up to 10^4 iterations
    T_random = 100000
    T_kuhn = 100000
    T_mat = 100000
    T_goofspiel = 100000

    # Dimensions chosen in run_algorithms.py. In the paper: n,m = 30,40
    n,m = 10,15

    fig, axes = plt.subplots(nrows=1, ncols=4, sharex=False, sharey=False, figsize=(43, 5))
    
    fig.text(0.5, -0.1, 'Number of iterations', ha='center',fontsize=40)
    fig.text(0.07, 0.5, 'Duality gap', va='center', rotation='vertical',fontsize=40)
    
    
    cols = ['3x3 matrix game','Kuhn Poker','Goofspiel',r'Random']
    
    for col in range(len(cols)):
        axes[col].set_title(cols[col],fontsize=40)
        
    ################################################################################
    ################################################
    ################################################    First row 
    
    ################ 0,0
    
    ### small matrix game instance
    
    col = 0
    
    nb_points_loc = 15
    x_axis = np.asarray(range(T_mat-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=35)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    
    result_duality_gap_rmp = np.load(f'numerical_results/rmp_duality_gap_{T_mat}_eta_{eta}.npy')        
    result_duality_gap_alt_rmp = np.load(f'numerical_results/alt_rmp_duality_gap_{T_mat}_eta_{eta}.npy')        
    result_duality_gap_prmp = np.load(f'numerical_results/prmp_duality_gap_{T_mat}_eta_{eta}.npy')   
    result_duality_gap_alt_prmp = np.load(f'numerical_results/alt_prmp_duality_gap_{T_mat}_eta_{eta}.npy')   
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T_mat}_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T_mat}_eta_{eta}.npy')  

    eta_loc = 0.05
    result_duality_gap_EG = np.load(f'numerical_results/extragradient_alg_duality_gap_{T_kuhn}_eta_{eta_loc}.npy')        
    result_duality_gap_OG = np.load(f'numerical_results/optimistic_omd_duality_gap_{T_kuhn}_eta_{eta_loc}.npy')     
    
        
    
    axes[col].plot(x_axis[indices], result_duality_gap_rmp[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_rmp[indices], color='yellow', marker='p', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_prmp[indices], color='blue', marker='D', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_prmp[indices], color='black', marker='d', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)
    
    
    axes[col].plot(x_axis[indices], result_duality_gap_EG[indices], color='brown', marker='>', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_OG[indices], color='pink', marker='o', linestyle=':',linewidth=2, markersize=10)


    if T_mat == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
        
    ################ 0,1

    ### Kuhn instance
    
    col = 1
    
    nb_points_loc = 15
    x_axis = np.asarray(range(T_kuhn-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=35)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    result_duality_gap_rmp = np.load(f'numerical_results/rmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_alt_rmp = np.load(f'numerical_results/alt_rmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_prmp = np.load(f'numerical_results/prmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')   
    result_duality_gap_alt_prmp = np.load(f'numerical_results/alt_prmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')   
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')     
    
    eta_loc = 0.01
    result_duality_gap_EG = np.load(f'numerical_results/extragradient_alg_duality_gap_{T_kuhn}_kuhn_eta_{eta_loc}.npy')   
    eta_loc = 0.1
    result_duality_gap_OG = np.load(f'numerical_results/optimistic_omd_duality_gap_{T_kuhn}_kuhn_eta_{eta_loc}.npy')   
    
    # threshold for duality gaps on plots
    threshold = 10e-17
    result_duality_gap_rmp = np.maximum(result_duality_gap_rmp, threshold)

    result_duality_gap_alt_rmp = np.maximum(result_duality_gap_alt_rmp, threshold)
    result_duality_gap_prmp = np.maximum(result_duality_gap_prmp, threshold)
    result_duality_gap_alt_prmp = np.maximum(result_duality_gap_alt_prmp, threshold)
    result_duality_gap_crmp = np.maximum(result_duality_gap_crmp, threshold)
    result_duality_gap_smooth_prmp = np.maximum(result_duality_gap_smooth_prmp, threshold)
    
    result_duality_gap_EG = np.maximum(result_duality_gap_EG, threshold)
    result_duality_gap_OG = np.maximum(result_duality_gap_OG, threshold)
    
    axes[col].plot(x_axis[indices], result_duality_gap_rmp[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_rmp[indices], color='yellow', marker='p', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_prmp[indices], color='blue', marker='D', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_prmp[indices], color='black', marker='d', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)
    
    axes[col].plot(x_axis[indices], result_duality_gap_EG[indices], color='brown', marker='>', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_OG[indices], color='pink', marker='o', linestyle=':',linewidth=2, markersize=10)


       
    if T_kuhn == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
        
    ### Goofspiel instance
    
    col = 2
    
    
    nb_points_loc = 15
    x_axis = np.asarray(range(T_kuhn-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=35)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    result_duality_gap_rmp = np.load(f'numerical_results/rmp_duality_gap_{T_goofspiel}_goofspiel.npy')        
    result_duality_gap_alt_rmp = np.load(f'numerical_results/alt_rmp_duality_gap_{T_goofspiel}_goofspiel.npy')        
    result_duality_gap_prmp = np.load(f'numerical_results/prmp_duality_gap_{T_goofspiel}_goofspiel.npy')   
    result_duality_gap_alt_prmp = np.load(f'numerical_results/alt_prmp_duality_gap_{T_goofspiel}_goofspiel.npy')   
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        
    
    eta_loc = 0.01
    result_duality_gap_EG = np.load(f'numerical_results/extragradient_alg_duality_gap_{T_kuhn}_goofspiel_eta_{eta_loc}.npy')   
    eta_loc = 0.001
    result_duality_gap_OG = np.load(f'numerical_results/optimistic_omd_duality_gap_{T_kuhn}_goofspiel_eta_{eta_loc}.npy')   
    
    
    # threshold for duality gaps on plots
    threshold = 10e-17
    result_duality_gap_rmp = np.maximum(result_duality_gap_rmp, threshold)

    result_duality_gap_alt_rmp = np.maximum(result_duality_gap_alt_rmp, threshold)
    result_duality_gap_prmp = np.maximum(result_duality_gap_prmp, threshold)
    result_duality_gap_alt_prmp = np.maximum(result_duality_gap_alt_prmp, threshold)
    result_duality_gap_crmp = np.maximum(result_duality_gap_crmp, threshold)
    result_duality_gap_smooth_prmp = np.maximum(result_duality_gap_smooth_prmp, threshold)
    
    result_duality_gap_EG = np.maximum(result_duality_gap_EG, threshold)
    result_duality_gap_OG = np.maximum(result_duality_gap_OG, threshold)
    
    
    axes[col].plot(x_axis[indices], result_duality_gap_rmp[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_rmp[indices], color='yellow', marker='p', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_prmp[indices], color='blue', marker='D', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_alt_prmp[indices], color='black', marker='d', linestyle='-',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)

    axes[col].plot(x_axis[indices], result_duality_gap_EG[indices], color='brown', marker='>', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_OG[indices], color='pink', marker='o', linestyle=':',linewidth=2, markersize=10)

       
    if T_goofspiel == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
    
    ################ 0,2
    
    ### Random instances
    
    T = T_random
    
    col=3
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=35)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    alg = 'rmp'
    result_duality_gap_rmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_RM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    alg = 'alt_rmp'
    result_duality_gap_alt_rmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_alt_RM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    alg = 'prmp'     
    result_duality_gap_prmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_PRM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    alg = 'alt_prmp'     
    result_duality_gap_alt_prmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_alt_PRM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    alg = 'crmp'
    result_duality_gap_crmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_CRM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    alg = 'smooth_prmp'
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')   
    y_PRM_plus_chop_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    
    alg = 'extragradient_alg'
    eta_loc = 0.1
    result_duality_gap_EG = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta_loc}.npy')   
    y_EG_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta_loc}_ci.npy')
    
    alg = 'optimistic_omd'
    eta_loc = 0.01
    result_duality_gap_OG = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta_loc}.npy')   
    y_OG_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta_loc}_ci.npy')       
    
    
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  

    axes[col].plot(x_axis[indices], result_duality_gap_rmp, color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label='RM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_alt_rmp, color='yellow', marker='p', linestyle='--',linewidth=2, markersize=10,label = 'alt. RM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_prmp, color='blue', marker='D', linestyle='-',linewidth=2, markersize=10,label='PRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_alt_prmp, color='black', marker='d', linestyle=':',linewidth=2, markersize=10,label = 'alt. PRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_crmp, color='green', marker='<', linestyle='--',linewidth=2, markersize=10,label='ExRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp, color='grey', marker='s', linestyle='-',linewidth=2, markersize=10,label='Smooth PRM$^+$')

    axes[col].plot(x_axis[indices], result_duality_gap_EG, color='brown', marker='>', linestyle='--',linewidth=2, markersize=10,label='EG')
    axes[col].plot(x_axis[indices], result_duality_gap_OG, color='pink', marker='o', linestyle=':',linewidth=2, markersize=10,label='OG')
        
    
    # threshold for duality gaps on plots
    threshold = 10e-17
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_rmp-y_RM_plus_ci,threshold),result_duality_gap_rmp+y_RM_plus_ci, color='red', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_alt_rmp-y_alt_RM_plus_ci,threshold),result_duality_gap_alt_rmp+y_alt_RM_plus_ci, color='yellow', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_prmp-y_PRM_plus_ci,threshold),result_duality_gap_prmp+y_PRM_plus_ci, color='blue', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_alt_prmp-y_alt_PRM_plus_ci,threshold),result_duality_gap_alt_prmp+y_alt_PRM_plus_ci, color='black', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_crmp-y_CRM_plus_ci,threshold),result_duality_gap_crmp+y_CRM_plus_ci, color='green', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_smooth_prmp-y_PRM_plus_chop_ci,threshold),result_duality_gap_smooth_prmp+y_PRM_plus_chop_ci, color='grey', alpha=0.1)

    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_EG-y_EG_ci,threshold),result_duality_gap_EG+y_EG_ci, color='brown', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_OG-y_OG_ci,threshold),result_duality_gap_OG+y_OG_ci, color='pink', alpha=0.1)


    if T_random == 10000:
        axes[col].set_xticks([10,100,1000,10000])
       
    if T_random == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])

    ################################################################################
    ################################################
    ################################################    Axis parameter
        
    
    fig.legend(ncol=8,loc="upper center",bbox_to_anchor=(0.5,1.3,0,0),fontsize=40)
    fig.savefig(f"figures/plots_with_EG_OG.pdf", bbox_inches='tight')
    fig.savefig(f"figures/plots_with_EG_OG.pgf", bbox_inches='tight')
    plt.show()

############## Fig 4-6: showing the last iterates of
# RM+, alt RM+, PRM+ on the small matrix instance
if plot_strategy_profile_small_matrix:

    results_x=np.load(f'numerical_results/{alg}_results_x_{T}_eta_{eta}.npy')        
    results_y=np.load(f'numerical_results/{alg}_results_y_{T}_eta_{eta}.npy')   
    
    axis_x_0=results_x[(len(results_x)-nb_trajectory):(len(results_x)-2),0]
    axis_x_1=results_x[(len(results_x)-nb_trajectory):(len(results_x)-2),1]
    
    fig, ax = plt.subplots()
    if alg == 'rmp':    
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = r'RM$^+$')
    elif alg == 'prmp':
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'Predictive RM$^+$')
    elif alg == 'alt_rmp':
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'Alternating RM$^+$')
    elif alg == 'alt_prmp':
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'alt. PRM$^+$')
    elif alg == 'crmp':
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'ExRM$^+$')
    elif alg == 'smooth_prmp':
        l1 =  ax.plot(axis_x_0, axis_x_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'SPRM$^+$')
    
    ax.set_xlabel(r'$x_{1}$',fontsize=20)
    ax.set_ylabel(r'$x_{2}$',fontsize=20)
    ax.tick_params(axis='both', which='major', labelsize=20)
    ax.set_xlim(max(min(axis_x_0)-0.005,0),max(axis_x_0)+0.005)
    ax.set_ylim(max(min(axis_x_1)-0.005,0),max(axis_x_1)+0.005)
    ax.grid()
    
    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.06,0,0),fontsize=20)


    # fig.savefig(f"figures/x_strategy_profile_T_{T}_{alg}.pgf", bbox_inches='tight')
    fig.savefig(f"figures/x_strategy_profile_T_{T}_{alg}.pdf", bbox_inches='tight')
    fig.savefig(f"figures/x_strategy_profile_T_{T}_{alg}.jpg", bbox_inches='tight')
    

    axis_y_0=results_y[(len(results_x)-nb_trajectory):(len(results_x)-2),0]
    axis_y_1=results_y[(len(results_x)-nb_trajectory):(len(results_x)-2),1]
    
    fig, ax = plt.subplots()
    if alg == 'rmp':    
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = r'RM$^+$')
    elif alg == 'prmp':
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'Predictive RM$^+$')
    elif alg == 'alt_rmp':
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'aAlternating RM$^+$')
    elif alg == 'alt_prmp':
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'alt. PRM$^+$')
    elif alg == 'crmp':
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'ExRM$^+$')
    elif alg == 'smooth_prmp':
        l1 =  ax.plot(axis_y_0, axis_y_1, color='red', marker='*', linestyle='-',linewidth=1, markersize=1,label = rf'SPRM$^+$')
                
    ax.set_xlabel(r'$y_{1}$',fontsize=20)
    ax.set_ylabel(r'$y_{2}$',fontsize=20)
    ax.tick_params(axis='both', which='major', labelsize=20)
    ax.set_xlim(max(min(axis_y_0)-0.005,0),max(axis_y_0)+0.005)
    ax.set_ylim(max(min(axis_y_1)-0.005,0),max(axis_y_1)+0.005)
    ax.grid()
    
    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.07,0,0),fontsize=20)


    fig.savefig(f"figures/y_strategy_profile_T_{T}_{alg}.pdf", bbox_inches='tight')
    fig.savefig(f"figures/y_strategy_profile_T_{T}_{alg}.jpg", bbox_inches='tight')

############## Fig 7: Duality gaps of
###   ExRM+, Smooth PRM+, RS-ExRM+, RS-Smooth PRM+
### over Kuhn, Goofspiel, small matrix, random instances:
if plot_both_matrix_instances_4_plots_with_restart:
    

    eta = 0.05

    
    T_random = 100000
    T = 100000
    T_kuhn = 100000
    T_goofspiel = 100000
    
    # Dimensions chosen in run_algorithms.py. In the paper: n,m = 30,40
    n,m = 10,15

    fig, axes = plt.subplots(nrows=1, ncols=4, sharex=False, sharey=False, figsize=(25, 5))
    
    fig.text(0.5, -0.05, 'Number of iterations', ha='center',fontsize=30)
    fig.text(0.07, 0.5, 'Duality gap', va='center', rotation='vertical',fontsize=30)
    
    
    cols = ['3x3 matrix game','Kuhn Poker','Goofspiel',r'random']
    
    for col in range(len(cols)):
        axes[col].set_title(cols[col],fontsize=25)
        
    ################################################################################
    ################################################
    ################################################    First row 
    
    ################ 0,0
    
    ### small matrix game instance
    
    col = 0
    
    nb_points_loc = 15
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=25)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T}_eta_{eta}.npy')        
    result_duality_gap_crmp_rs = np.load(f'numerical_results/crmp_rs_duality_gap_{T}_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T}_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp_rs = np.load(f'numerical_results/smooth_prmp_rs_duality_gap_{T}_eta_{eta}.npy')        
    

    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp_rs[indices], color='red', marker='p', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp_rs[indices], color='blue', marker='*', linestyle=':',linewidth=2, markersize=10)
   
    if T == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
    ################ 0,1

    ### Kuhn instance
    
    col = 1
    

    nb_points_loc = 15
    x_axis = np.asarray(range(T_kuhn-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=25)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_crmp_rs = np.load(f'numerical_results/crmp_rs_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp_rs = np.load(f'numerical_results/smooth_prmp_rs_duality_gap_{T_kuhn}_kuhn_eta_{eta}.npy')        
  
    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp_rs[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp_rs[indices], color='blue', marker='p', linestyle='--',linewidth=2, markersize=10)
    
    if T == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
        
    ### Goofspiel instance
    
    col = 2
    

    nb_points_loc = 15
    x_axis = np.asarray(range(T_kuhn-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points_loc)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=25)
    axes[col].set_yscale('log')
    axes[col].set_xscale('log')
    
    
    result_duality_gap_crmp = np.load(f'numerical_results/crmp_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        
    result_duality_gap_crmp_rs = np.load(f'numerical_results/crmp_rs_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/smooth_prmp_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        
    result_duality_gap_smooth_prmp_rs = np.load(f'numerical_results/smooth_prmp_rs_duality_gap_{T_goofspiel}_goofspiel_eta_{eta}.npy')        

  
    axes[col].plot(x_axis[indices], result_duality_gap_crmp[indices], color='green', marker='<', linestyle='--',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp[indices], color='grey', marker='s', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_crmp_rs[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10)
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp_rs[indices], color='blue', marker='p', linestyle='--',linewidth=2, markersize=10)
    
    if T == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])
        
    ################ 0,2
    
    ### Random instances
    
    T = T_random
    
    col=3
    
    axes[col].grid()
    
    axes[col].tick_params(labelsize=25)
    axes[col].set_yscale('log')
    # axes[col].set_yscale('symlog',linthresh = 10e-10) 
    axes[col].set_xscale('log')
    

    alg = 'crmp'
    result_duality_gap_crmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_CRM_plus_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')

    alg = 'crmp_rs'
    result_duality_gap_crmp_rs = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')  
    y_CRM_plus_rs_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
    
    alg = 'smooth_prmp'
    result_duality_gap_smooth_prmp = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')   
    y_PRM_plus_chop_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
   
    alg = 'smooth_prmp_rs'
    result_duality_gap_smooth_prmp_rs = np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy')   
    y_PRM_plus_chop_rs_ci=np.load(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy')
        
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2  

    axes[col].plot(x_axis[indices], result_duality_gap_crmp, color='green', marker='<', linestyle='--',linewidth=2, markersize=10,label='ExRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp, color='grey', marker='s', linestyle='-',linewidth=2, markersize=10,label='Smooth PRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_crmp_rs, color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label='RS-ExRM$^+$')
    axes[col].plot(x_axis[indices], result_duality_gap_smooth_prmp_rs, color='blue', marker='p', linestyle='--',linewidth=2, markersize=10,label = 'RS-Smooth PRM$^+$')
       
    # threshold for duality gaps on plots
    threshold = 10e-14

    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_crmp-y_CRM_plus_ci,threshold),result_duality_gap_crmp+y_CRM_plus_ci, color='green', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_smooth_prmp-y_PRM_plus_chop_ci,threshold),result_duality_gap_smooth_prmp+y_PRM_plus_chop_ci, color='grey', alpha=0.1)    
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_crmp_rs-y_CRM_plus_rs_ci,threshold),result_duality_gap_crmp+y_CRM_plus_rs_ci, color='red', alpha=0.1)
    axes[col].fill_between(x_axis[indices], np.maximum(result_duality_gap_smooth_prmp_rs-y_PRM_plus_chop_rs_ci,threshold),result_duality_gap_smooth_prmp_rs+y_PRM_plus_chop_rs_ci, color='blue', alpha=0.1)

    if T_random == 100000:
        axes[col].set_xticks([10,100,1000,10000,100000])

    ################################################################################
    ################################################
    ################################################    Axis parameter
        
    
    fig.legend(ncol=6,loc="upper center",bbox_to_anchor=(0.5,1.13,0,0),fontsize=25)
    
    fig.savefig(f"figures/small_matrix_{T}_and_random_{T_random}_with_restart_with_Kuhn_Goofspiel_eta_{eta}.pgf", bbox_inches='tight')
    fig.savefig(f"figures/small_matrix_{T}_and_random_{T_random}_with_restart_with_Kuhn_Goofspiel_eta_{eta}.pdf", bbox_inches='tight')
    plt.show()


